﻿/*
	VERSION:		1.5
	1.5		Now uses avoidLoopPanic() to (hopefully) prevent very long scripts from freezing Flash.
	1.4		added abort() to the resulting array, which can be called manually to abort the script
	1.3		done() only fires one time per script-run.  done() can be safely evoked even if eventIndex is far beyond the length.
	1.2		Added "onScriptDone" event
	1.1		Added an option to define checkForInterrupt() that can interrupt the script when nextEvent() is called
	
	ABOUT THIS:
		This is a modified version of scriptSystem.as version 1.5
		It stores an array of objects, each containing:
			run()				This function is called when this command is run
			command			This object contains command data, and is passed to the run() function.
			baseScript	This object contains the root script.  It's used by commands to set the "timeoutID"
	
	PURPOSE:
		This runs blocks of code in sequence, as a chain-reaction.
		allowing for arbitrary delays and halts.
		This is simple but very powerful.
		
	USAGE:
		talk = makeScriptSystem();
		talk[0] = {
			run: function( self_obj ){},
			myVar1: 42,
			myVar2: "merf"
		}
		talk.runEvent(0);
		talk.done = function(){}
		// automatically used by the script to determine whether or not it's allowed to continue after each command
		talk.checkForInterrupt = function(){
			// if sprite doesn't exist
			return false;		// prevents this script from running
		}
		
	NOTE:
		A standard set of event functions can be defined,
		which makes it easier to use this because 
		they can automatically call nextEvent() afterwords.
		example:
			events.push(		talk( events, "Hello World!");		);
		A reference to the script system is passed to them,
		allowing them to resume the script when they're done.
		
	FUNCTIONS:
		done()				Externally-defined function that's called when the script reaches its end.
		
	EVENTS:
		onScriptDone	Fires after the script has finished executing
		
	PROPERTIES:
		isDone				Indicates whether the script has already finished running.		(null=never ran  false=running  true=complete)
*/
#include "avoidLoopPanic.as"
makeScriptSystem = function()
{
	// create container
	var _this = [];
	
	// add events
	AsBroadcaster.initialize(_this);
	
	// variables
	_this.index = 0;				// current event index
	_this.isDone = null;		// script has never been run
	
	// prevent Flash from panicking when a ton of script-commands are called at once
	if(!_global.AVOID_LOOP_PANIC){
		_global.AVOID_LOOP_PANIC = make_avoidLoopPanic( 100 );
	}
	
	
	
	// ______________________________________________________________________________________________________
	// FUNCTIONS
	var done = function(){
		// avoid completing the same run twice
		if(_this.isDone === true)		return;
		
		_this.done();										// announce that this script has just completed
		_this.broadcastMessage("onScriptDone");
		// prevent nextEvent() from being able to run anything, until this script is explicitly started again
		_this.index = _this.length;
		_this.isDone = true;							// script IS done
	}// done()
	_this.abort = done;
	
	
	_this.runEvent = function( eventIndex ){
		//AVOID_LOOP_PANIC(function(){
															
			var eventIndex = eventIndex || 0;			// if no event is specified, start from the beginning
			_this.index = eventIndex;							// remember where you are in the list
			
			if(eventIndex >= _this.length)
			{// if:  Trying to call an event that doesn't exist
				done();
			}else{
				_this.isDone = false;							// script isn't done
				_this[eventIndex].run( _this[eventIndex] );		// pass this command's object, containing:  data,  run(),  nextEvent(), storeTimeout()
			}
			
		//});// AVOID_LOOP_PANIC()
	}// runEvent()
	
	
	
	_this.nextEvent = function(){
		AVOID_LOOP_PANIC(function(){
															
			var allowRun = (_this.checkForInterrupt) ? _this.checkForInterrupt() : true;
			if(allowRun){		// if:  allowed to continue
				_this.runEvent(_this.index+1);		// this function also updates "index"
			}else{						// if:  told not to continue
				done();
			}
			
		});// AVOID_LOOP_PANIC()
	}// nextEvent()
	// ______________________________________________________________________________________________________
	
	
	
	// return container
	return _this;
}// makeScriptSystem()